/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.CountDown;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.env.Environment;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.xpack.core.security.authc.Realm;
import org.elasticsearch.xpack.core.security.authc.RealmConfig;
import org.elasticsearch.xpack.core.security.authc.RealmSettings;
import org.elasticsearch.xpack.security.authc.InternalRealms;
import org.elasticsearch.xpack.security.authc.esnative.ReservedRealm;

public class Realms
implements Iterable<Realm> {
    private static final Logger logger = LogManager.getLogger(Realms.class);
    private static final DeprecationLogger deprecationLogger = new DeprecationLogger(logger);
    private final Settings settings;
    private final Environment env;
    private final Map<String, Realm.Factory> factories;
    private final XPackLicenseState licenseState;
    private final ThreadContext threadContext;
    private final ReservedRealm reservedRealm;
    protected List<Realm> realms;
    List<Realm> standardRealmsOnly;
    List<Realm> nativeRealmsOnly;

    public Realms(Settings settings, Environment env, Map<String, Realm.Factory> factories, XPackLicenseState licenseState, ThreadContext threadContext, ReservedRealm reservedRealm) throws Exception {
        this.settings = settings;
        this.env = env;
        this.factories = factories;
        this.licenseState = licenseState;
        this.threadContext = threadContext;
        this.reservedRealm = reservedRealm;
        assert (factories.get("reserved") == null);
        this.realms = this.initRealms();
        ArrayList<Realm> standardRealms = new ArrayList<Realm>();
        ArrayList<Realm> nativeRealms = new ArrayList<Realm>();
        for (Realm realm : this.realms) {
            if (InternalRealms.isStandardRealm(realm.type())) {
                standardRealms.add(realm);
            }
            if (!"file".equals(realm.type()) && !"native".equals(realm.type())) continue;
            nativeRealms.add(realm);
        }
        for (List realmList : Arrays.asList(standardRealms, nativeRealms)) {
            if (realmList.isEmpty()) {
                this.addNativeRealms(realmList);
            }
            assert (!realmList.contains((Object)reservedRealm));
            realmList.add(0, reservedRealm);
            assert (realmList.get(0) == reservedRealm);
        }
        this.standardRealmsOnly = Collections.unmodifiableList(standardRealms);
        this.nativeRealmsOnly = Collections.unmodifiableList(nativeRealms);
        this.realms.forEach(r -> r.initialize((Iterable)this, licenseState));
    }

    @Override
    public Iterator<Realm> iterator() {
        return this.asList().iterator();
    }

    public List<Realm> getUnlicensedRealms() {
        XPackLicenseState licenseStateSnapshot = this.licenseState.copyCurrentLicenseState();
        if (!licenseStateSnapshot.isSecurityEnabled()) {
            return Collections.unmodifiableList(this.realms);
        }
        if (licenseStateSnapshot.checkFeature(XPackLicenseState.Feature.SECURITY_ALL_REALMS)) {
            return Collections.emptyList();
        }
        List<Realm> allowedRealms = this.asList();
        if (allowedRealms.equals(this.realms)) {
            return Collections.emptyList();
        }
        List unlicensed = this.realms.stream().filter(r -> !allowedRealms.contains(r)).collect(Collectors.toList());
        return Collections.unmodifiableList(unlicensed);
    }

    public Stream<Realm> stream() {
        return StreamSupport.stream(this.spliterator(), false);
    }

    public List<Realm> asList() {
        XPackLicenseState licenseStateSnapshot = this.licenseState.copyCurrentLicenseState();
        if (!licenseStateSnapshot.isSecurityEnabled()) {
            return Collections.emptyList();
        }
        if (licenseStateSnapshot.checkFeature(XPackLicenseState.Feature.SECURITY_ALL_REALMS)) {
            return this.realms;
        }
        if (licenseStateSnapshot.checkFeature(XPackLicenseState.Feature.SECURITY_STANDARD_REALMS)) {
            return this.standardRealmsOnly;
        }
        return this.nativeRealmsOnly;
    }

    public Realm realm(String name) {
        for (Realm realm : this.realms) {
            if (!name.equals(realm.name())) continue;
            return realm;
        }
        return null;
    }

    public Realm.Factory realmFactory(String type) {
        return this.factories.get(type);
    }

    protected List<Realm> initRealms() throws Exception {
        Map realmsSettings = RealmSettings.getRealmSettings((Settings)this.settings);
        HashSet<String> internalTypes = new HashSet<String>();
        ArrayList<Realm> realms = new ArrayList<Realm>();
        ArrayList<String> kerberosRealmNames = new ArrayList<String>();
        HashMap<String, Set> nameToRealmIdentifier = new HashMap<String, Set>();
        TreeSet<String> missingOrderRealmSettingKeys = new TreeSet<String>();
        HashMap<String, Set<String>> orderToRealmOrderSettingKeys = new HashMap<String, Set<String>>();
        for (Map.Entry entry2 : realmsSettings.entrySet()) {
            RealmConfig.RealmIdentifier identifier = (RealmConfig.RealmIdentifier)entry2.getKey();
            if (!((Settings)entry2.getValue()).hasValue("order")) {
                missingOrderRealmSettingKeys.add(RealmSettings.getFullSettingKey((RealmConfig.RealmIdentifier)identifier, (Function)RealmSettings.ORDER_SETTING));
            } else {
                orderToRealmOrderSettingKeys.computeIfAbsent(((Settings)entry2.getValue()).get("order"), k -> new TreeSet()).add(RealmSettings.getFullSettingKey((RealmConfig.RealmIdentifier)identifier, (Function)RealmSettings.ORDER_SETTING));
            }
            Realm.Factory factory = this.factories.get(identifier.getType());
            if (factory == null) {
                throw new IllegalArgumentException("unknown realm type [" + identifier.getType() + "] for realm [" + identifier + "]");
            }
            RealmConfig config = new RealmConfig(identifier, this.settings, this.env, this.threadContext);
            if (!config.enabled()) {
                if (!logger.isDebugEnabled()) continue;
                logger.debug("realm [{}] is disabled", (Object)identifier);
                continue;
            }
            if ("file".equals(identifier.getType()) || "native".equals(identifier.getType())) {
                if (internalTypes.contains(identifier.getType())) {
                    throw new IllegalArgumentException("multiple [" + identifier.getType() + "] realms are configured. [" + identifier.getType() + "] is an internal realm and therefore there can only be one such realm configured");
                }
                internalTypes.add(identifier.getType());
            }
            if ("kerberos".equals(identifier.getType())) {
                kerberosRealmNames.add(identifier.getName());
                if (kerberosRealmNames.size() > 1) {
                    throw new IllegalArgumentException("multiple realms " + ((Object)kerberosRealmNames).toString() + " configured of type [" + identifier.getType() + "], [" + identifier.getType() + "] can only have one such realm configured");
                }
            }
            Realm realm = factory.create(config);
            nameToRealmIdentifier.computeIfAbsent(realm.name(), k -> new HashSet()).add(RealmSettings.realmSettingPrefix((String)realm.type()) + realm.name());
            realms.add(realm);
        }
        if (!realms.isEmpty()) {
            Collections.sort(realms);
        } else {
            this.addNativeRealms(realms);
        }
        realms.add(0, this.reservedRealm);
        String duplicateRealms = nameToRealmIdentifier.entrySet().stream().filter(entry -> ((Set)entry.getValue()).size() > 1).map(entry -> (String)entry.getKey() + ": " + entry.getValue()).collect(Collectors.joining("; "));
        if (Strings.hasText((String)duplicateRealms)) {
            throw new IllegalArgumentException("Found multiple realms configured with the same name: " + duplicateRealms + "");
        }
        this.logDeprecationIfFound(missingOrderRealmSettingKeys, orderToRealmOrderSettingKeys);
        return Collections.unmodifiableList(realms);
    }

    public void usageStats(ActionListener<Map<String, Object>> listener) {
        XPackLicenseState licenseStateSnapshot = this.licenseState.copyCurrentLicenseState();
        HashMap realmMap = new HashMap();
        AtomicBoolean failed = new AtomicBoolean(false);
        List realmList = this.asList().stream().filter(r -> !"reserved".equals(r.type())).collect(Collectors.toList());
        Set realmTypes = realmList.stream().map(Realm::type).collect(Collectors.toSet());
        CountDown countDown = new CountDown(realmList.size());
        Runnable doCountDown = () -> {
            if ((realmList.isEmpty() || countDown.countDown()) && !failed.get()) {
                for (String type : this.factories.keySet()) {
                    assert (!"reserved".equals(type));
                    realmMap.compute(type, (key, value) -> {
                        if (value == null) {
                            return MapBuilder.newMapBuilder().put((Object)"enabled", (Object)false).put((Object)"available", (Object)Realms.isRealmTypeAvailable(licenseStateSnapshot, type)).map();
                        }
                        assert (value instanceof Map);
                        Map realmTypeUsage = (Map)value;
                        realmTypeUsage.put("enabled", true);
                        realmTypeUsage.put("available", true);
                        return value;
                    });
                }
                listener.onResponse((Object)realmMap);
            }
        };
        if (realmList.isEmpty()) {
            doCountDown.run();
        } else {
            for (Realm realm : realmList) {
                realm.usageStats(ActionListener.wrap(stats -> {
                    if (!failed.get()) {
                        Map map = realmMap;
                        synchronized (map) {
                            realmMap.compute(realm.type(), (key, value) -> {
                                if (value == null) {
                                    Map<String, Object> realmTypeUsage = Realms.convertToMapOfLists(stats);
                                    return realmTypeUsage;
                                }
                                assert (value instanceof Map);
                                Realms.combineMaps((Map)value, stats);
                                return value;
                            });
                        }
                        doCountDown.run();
                    }
                }, e -> {
                    if (failed.compareAndSet(false, true)) {
                        listener.onFailure(e);
                    }
                }));
            }
        }
    }

    private void addNativeRealms(List<Realm> realms) throws Exception {
        Realm.Factory indexRealmFactory;
        Realm.Factory fileRealm = this.factories.get("file");
        if (fileRealm != null) {
            realms.add(fileRealm.create(new RealmConfig(new RealmConfig.RealmIdentifier("file", "default_file"), this.settings, this.env, this.threadContext)));
        }
        if ((indexRealmFactory = this.factories.get("native")) != null) {
            realms.add(indexRealmFactory.create(new RealmConfig(new RealmConfig.RealmIdentifier("native", "default_native"), this.settings, this.env, this.threadContext)));
        }
    }

    private static void combineMaps(Map<String, Object> mapA, Map<String, Object> mapB) {
        for (Map.Entry<String, Object> entry : mapB.entrySet()) {
            mapA.compute(entry.getKey(), (key, value) -> {
                if (value == null) {
                    return new ArrayList(Collections.singletonList(entry.getValue()));
                }
                assert (value instanceof List);
                ((List)value).add(entry.getValue());
                return value;
            });
        }
    }

    private static Map<String, Object> convertToMapOfLists(Map<String, Object> map) {
        HashMap<String, Object> converted = new HashMap<String, Object>(map.size());
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            converted.put(entry.getKey(), new ArrayList<Object>(Collections.singletonList(entry.getValue())));
        }
        return converted;
    }

    public static boolean isRealmTypeAvailable(XPackLicenseState licenseState, String type) {
        if (licenseState.checkFeature(XPackLicenseState.Feature.SECURITY_ALL_REALMS)) {
            return true;
        }
        if (licenseState.checkFeature(XPackLicenseState.Feature.SECURITY_STANDARD_REALMS)) {
            return InternalRealms.isStandardRealm(type) || "reserved".equals(type);
        }
        return "file".equals(type) || "native".equals(type);
    }

    private void logDeprecationIfFound(Set<String> missingOrderRealmSettingKeys, Map<String, Set<String>> orderToRealmOrderSettingKeys) {
        List duplicatedRealmOrderSettingKeys;
        if (missingOrderRealmSettingKeys.size() > 0) {
            deprecationLogger.deprecatedAndMaybeLog("unordered_realm_config", "Found realms without order config: [{}]. In next major release, node will fail to start with missing realm order.", new Object[]{String.join((CharSequence)"; ", missingOrderRealmSettingKeys)});
        }
        if (!(duplicatedRealmOrderSettingKeys = orderToRealmOrderSettingKeys.entrySet().stream().filter(e -> ((Set)e.getValue()).size() > 1).map(e -> (String)e.getKey() + ": " + String.join((CharSequence)",", (Iterable)e.getValue())).sorted().collect(Collectors.toList())).isEmpty()) {
            deprecationLogger.deprecatedAndMaybeLog("duplicate_realm_order", "Found multiple realms configured with the same order: [{}]. In next major release, node will fail to start with duplicated realm order.", new Object[]{String.join((CharSequence)"; ", duplicatedRealmOrderSettingKeys)});
        }
    }
}

